﻿/*
	VERSION:		2.2
	
	DESCRIPTION:
		This is a movieClip-based music system.
		It streams external .mp3 files.
	
	MINIMAL USAGE:
		#include "makeStreamingMusic.as"
		MUSIC = makeStreamingMusic();
		MUSIC.play( "songName" );
		
		
	MAX USAGE:
		#include "makeStreamingMusic.as"
		MUSIC = makeStreamingMusic( target_mc:MovieClip, newName:String, [newDepth:Number] );
		MUSIC.play( "songName" );
		
		
	FUNCTIONS:
		play( "songName" );						// Plays the songName_intro tune, then switches to songName (which should loop)
		playLoop( "songName" );				// Skip any intro's and specifically play the main loo pof the song
		stopMusic();								// Stops the music and forgets the last song
		fadeTo( newVolume, seconds );		// Fades the song volume to a new level
		
	PROPERTIES:
		volume								// scalar (percentage) that gets or sets the maximum playback volume of all music
		songVolume							// gets or sets the intensity of THIS song  (used for fades)
		
		Playback volume  =  songVolume * (volume/100)
		To change the volume of all music played, use "volume".
		To change the volume of only the current song, use "songVolume".
		All songs start playback with an "songVolume" of 100.
		
	INTROS & LOOPS:
		You may add folders before the fileNames.
		Name your files / linkages like so:
			songName_intro.mp3			play("songName.mp3")				(The extension must be .mp3)
			songName.mp3					playLoop("songName.mp3")			(This is the looping version of the song.)
		When play() is used, it'll automatically attempt to load an intro for the song. If there's no intro, it'll play the loop.
*/



makeStreamingMusic = function( target_mc, newName, newDepth )
{
	// resolve optional parameters
	var target_mc = (target_mc != undefined) ? target_mc : this;
	var newName = (newName != undefined) ? newName : "musicSystem_mc";
	#include "nextDepth.as"
	var newDepth = (newDepth) ? newDepth : nextDepth(target_mc);
	
	// create containers
	var _this = target_mc.createEmptyMovieClip( newName, newDepth );
	_this.music = new Sound();
	_this.musicVolume = 100;
	_this.isPlaying = false;
	_this.songName = "";
	
	
	
	
	
	
	// FUNCTIONS
	_this.play = function( songName )
	{
		if(_this.songName != songName)
		{
			// transition:  intro  ->  loop
			_this.music.onSoundComplete = function()
			{
				// play normally
				var songName = _this.songName;
				_this.songName = "";					// spoof it so it'll play the same song again
				var readSongVolume = _this.songVolume;		// remember intro song's volume
				_this.playLoop( songName );		// play the regular (looping) version
				_this.songVolume = readSongVolume;			// apply intro song's volume to looping song
			}// onSoundComplete()
			
			// If there is no intro,  immediately play the loop
			_this.music.onLoad = function(success)
			{
				if(!success)
					_this.music.onSoundComplete();
			}// songLoaded()
			
			var nameLength = songName.indexOf(".");
			var introName = songName.substr(0, nameLength) + "_intro" + ".mp3";		// songName.mp3  ->  songName_intro.mp3
			_this.songName = songName;
			_this.isPlaying = false;
			_this.music.loadSound( introName, true );
		}// if: a new song is specified
		
		if( _this.isPlaying == false )
		{
			_this.music.start();
			_this.volume = _this.musicVolume;
			_this.isPlaying = true;
		}// if:  music is not playing
	}// play()
	
	
	
	_this.playLoop = function( songName )
	{
		if(_this.songName != songName)
		{
			// Looping method
			_this.music.onSoundComplete = function()
			{
				_this.music.start();
			}// onSoundComplete()
			
			// Remove the no-intro detection
			_this.music.onLoad = function(){}
			
			_this.music.loadSound( songName, true );
			_this.songName = songName;
			_this.isPlaying = false;
		}// if: a new song is specified
		
		if( _this.isPlaying == false )
		{
			//_this.music.stop();		// might be redundant
			_this.music.start();
			_this.volume = _this.musicVolume;
			_this.isPlaying = true;
		}// if:  music is not playing
	}// playLoop()
	
	
	
	/*
	// Alternative looping method
	_this.checkLoop = function()
	{
		var current = _this.music.position;
		var end = _this.music.duration;
		if( current >= end-0 )
		{
			//_this.music.stop();
			_this.music.start();
		}
	}// checkLoop()
	_this.checkLoopInterval = setInterval( _this.checkLoop , 1 );
	_this.onUnload = function()
	{
		clearInterval( _this.checkLoopInterval );
	}// onUnload()
	*/
	
	
	
	_this.stopMusic = function()
	{
		_this.songName = "";
		_this.music.stop();
		_this.isPlaying = false;
	}// stopMusic()
	
	
	
	// getVolume()
	_this.getVolume = function()
	{
		return _this.musicVolume;
	}// volume		(get)
	// setVolume()
	_this.setVolume = function( newVolume )
	{
		newVolume = (newVolume==0) ? 1 : newVolume;		// minimum = 1;
		var thisSongVolume = _this.songVolume;
		_this.musicVolume = newVolume;
		var rawVolume = thisSongVolume * (_this.musicVolume/100);		// songVolume * global music volume
		_this.music.setVolume( rawVolume );
	}// volume		(set)
	_this.addProperty( "volume", _this.getVolume, _this.setVolume );
	
	
	
	_this.getSongVolume = function()
	{
		return _this.music.getVolume() / (_this.musicVolume/100);		// song volume  (global volume factored out);
	}// volume		(get)
	_this.setSongVolume = function( newVolume )
	{
		var rawVolume = newVolume * (_this.musicVolume/100);		// songVolume * global music volume
		_this.music.setVolume( rawVolume );
	}// volume		(set)
	_this.addProperty( "songVolume", _this.getSongVolume, _this.setSongVolume );
	
	
	
	// stop()
	_this.stop = _this.stopMusic;
	
	
	
	_this.fadeTo = function( endVolume, seconds )
	{
		// tween volume from current to target
		var startVolume = _this.getSongVolume();
		delete _this.fade;		// cancel previous fade
		_this.fade = new mx.transitions.Tween( _this, "songVolume", null, startVolume, endVolume, seconds, true);
		
		// onChange
		_this.fade.onMotionChanged = function()
		{
			_this.setSongVolume( this.position );
		}// onMotionChanged()
		
		// onDone
		_this.fade.onMotionFinished = function()
		{
			delete _this.fade;
			if(_this.songVolume == 0)
				_this.stopMusic();
		}// onMotionFinished()
	}// fadeTo()
	
	
	return _this;
}// makeStreamingMusic()